/**
 * StatusPage Widget Block Edit Component
 *
 * @package StatusPageWidget
 */

import { __ } from '@wordpress/i18n';
import { useBlockProps, InspectorControls } from '@wordpress/block-editor';
import {
    PanelBody,
    TextControl,
    SelectControl,
    ToggleControl,
    RangeControl,
    Placeholder,
    Spinner,
    Notice,
} from '@wordpress/components';
import { useState, useEffect, useCallback } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';

/**
 * Validate a slug format.
 * Only allows lowercase letters (a-z), numbers (0-9), and single hyphens between characters.
 *
 * @param {string} slug The slug to validate.
 * @return {boolean} Whether the slug is valid.
 */
function isValidSlug(slug) {
    if (!slug) return true; // Empty is ok during editing
    return /^[a-z0-9]+(-[a-z0-9]+)*$/.test(slug.toLowerCase());
}

/**
 * Get the status page URL from a slug.
 *
 * @param {string} slug The status page slug.
 * @return {string} The full URL.
 */
function getStatusPageUrl(slug) {
    return `https://${slug}.statuspage.me`;
}

/**
 * Edit component for the StatusPage Widget block.
 *
 * @param {Object}   props               Block props.
 * @param {Object}   props.attributes    Block attributes.
 * @param {Function} props.setAttributes Function to update attributes.
 * @return {JSX.Element} Block editor component.
 */
export default function Edit({ attributes, setAttributes }) {
    const {
        slug,
        theme,
        compact,
        showIncidents,
        rtm,
        upd,
        refresh,
        disableLink,
        renderMode,
    } = attributes;

    const [isLoading, setIsLoading] = useState(false);
    const [previewData, setPreviewData] = useState(null);
    const [error, setError] = useState(null);
    const [slugError, setSlugError] = useState(null);

    // Get defaults from localized data
    const defaults = window.statuspageWidgetData?.defaults || {};

    // Apply defaults on first render if attributes are at their initial values
    useEffect(() => {
        const updates = {};

        // Only apply defaults if the attribute is at its block.json default value
        if (!slug && defaults.slug) {
            updates.slug = defaults.slug;
        }
        if (theme === 'auto' && defaults.theme && defaults.theme !== 'auto') {
            updates.theme = defaults.theme;
        }
        if (compact === false && defaults.compact === true) {
            updates.compact = defaults.compact;
        }
        if (showIncidents === false && defaults.showIncidents === true) {
            updates.showIncidents = defaults.showIncidents;
        }
        if (rtm === 60 && defaults.rtm !== undefined && defaults.rtm !== 60) {
            updates.rtm = defaults.rtm;
        }
        if (upd === 7 && defaults.upd !== undefined && defaults.upd !== 7) {
            updates.upd = defaults.upd;
        }
        if (refresh === 0 && defaults.refresh && defaults.refresh !== 0) {
            updates.refresh = defaults.refresh;
        }
        if (disableLink === false && defaults.disableLink === true) {
            updates.disableLink = defaults.disableLink;
        }
        if (renderMode === 'js' && defaults.renderMode && defaults.renderMode !== 'js') {
            updates.renderMode = defaults.renderMode;
        }

        if (Object.keys(updates).length > 0) {
            setAttributes(updates);
        }
    }, []); // Run only once on mount

    const blockProps = useBlockProps();

    // Validate slug on change
    const handleSlugChange = (value) => {
        const lowercaseValue = value.toLowerCase();
        setAttributes({ slug: lowercaseValue });

        if (value && !isValidSlug(value)) {
            setSlugError(__('Invalid format. Use only lowercase letters, numbers, and single hyphens between characters.', 'statuspage-widget'));
        } else {
            setSlugError(null);
        }
    };

    // Fetch preview data for the editor
    const fetchPreview = useCallback(async () => {
        if (!slug || !isValidSlug(slug)) {
            setPreviewData(null);
            if (slug && !isValidSlug(slug)) {
                setError(__('Invalid slug format.', 'statuspage-widget'));
            } else {
                setError(null);
            }
            return;
        }

        setIsLoading(true);
        setError(null);

        try {
            const response = await fetch(
                `${window.statuspageWidgetData?.apiBase || 'https://statuspage.me'}/api/embed/status?slug=${encodeURIComponent(slug)}&inc=${showIncidents ? 1 : 0}&rtm=${rtm}&upd=${upd}`
            );

            if (!response.ok) {
                throw new Error(__('Unable to fetch status page data.', 'statuspage-widget'));
            }

            const data = await response.json();
            setPreviewData(data);
        } catch (err) {
            setError(err.message);
            setPreviewData(null);
        } finally {
            setIsLoading(false);
        }
    }, [slug, showIncidents, rtm, upd]);

    useEffect(() => {
        const timeoutId = setTimeout(fetchPreview, 500);
        return () => clearTimeout(timeoutId);
    }, [fetchPreview]);

    // Theme options
    const themeOptions = [
        { label: __('Auto (System)', 'statuspage-widget'), value: 'auto' },
        { label: __('Light', 'statuspage-widget'), value: 'light' },
        { label: __('Dark', 'statuspage-widget'), value: 'dark' },
    ];

    // Render mode options
    const renderModeOptions = [
        { label: __('JavaScript (recommended)', 'statuspage-widget'), value: 'js' },
        { label: __('PHP (server-side)', 'statuspage-widget'), value: 'php' },
    ];

    // Status colors
    const statusColors = {
        operational: '#16c172',
        degraded: '#ffb400',
        'partial-outage': '#ff7043',
        'major-outage': '#e53935',
        maintenance: '#3870ff',
        unknown: '#8c959f',
    };

    // Render preview widget in editor
    const renderPreview = () => {
        if (!slug) {
            return (
                <Placeholder
                    icon="admin-site-alt3"
                    label={__('StatusPage Widget', 'statuspage-widget')}
                    instructions={__(
                        'Enter your StatusPage.me slug in the block settings to display a status widget.',
                        'statuspage-widget'
                    )}
                >
                    <TextControl
                        label={__('Status Page Slug', 'statuspage-widget')}
                        value={slug}
                        onChange={handleSlugChange}
                        placeholder="your-status-page"
                    />
                    {slugError && (
                        <Notice status="warning" isDismissible={false}>
                            {slugError}
                        </Notice>
                    )}
                </Placeholder>
            );
        }

        if (isLoading) {
            return (
                <div className="statuspage-widget-loading">
                    <Spinner />
                    <span>{__('Loading status...', 'statuspage-widget')}</span>
                </div>
            );
        }

        if (error) {
            return (
                <Notice status="error" isDismissible={false}>
                    {error}
                </Notice>
            );
        }

        if (renderMode === 'php') {
            return (
                <ServerSideRender
                    block="statuspage-widget/status-widget"
                    attributes={attributes}
                />
            );
        }

        // JS mode preview
        if (previewData) {
            const overall = previewData.overall || { code: 'unknown', label: 'Status Unknown' };
            const statusColor = statusColors[overall.code] || statusColors.unknown;

            return (
                <div className={`spw-preview ${compact ? 'spw-compact' : ''} ${theme === 'dark' ? 'spw-dark' : ''}`}>
                    <div className="spw-preview-wrap">
                        <div className="spw-preview-inner">
                            <div className="spw-preview-header">
                                <span
                                    className="spw-preview-dot"
                                    style={{ backgroundColor: statusColor }}
                                />
                                <span className="spw-preview-label">{overall.label}</span>
                            </div>
                            {!compact && previewData.updated && (
                                <div className="spw-preview-meta">
                                    {__('Updated', 'statuspage-widget')} {previewData.updated.replace('T', ' ').replace('Z', ' UTC')}
                                </div>
                            )}
                            {!compact && showIncidents && previewData.incidents?.length > 0 && (
                                <ul className="spw-preview-incidents">
                                    {previewData.incidents.slice(0, 3).map((incident, i) => (
                                        <li key={i}>{incident.title}</li>
                                    ))}
                                    {previewData.incidents.length > 3 && (
                                        <li className="spw-muted">+{previewData.incidents.length - 3} more</li>
                                    )}
                                </ul>
                            )}
                        </div>
                    </div>
                </div>
            );
        }

        return null;
    };

    return (
        <>
            <InspectorControls>
                <PanelBody title={__('Status Page Settings', 'statuspage-widget')}>
                    <TextControl
                        label={__('Status Page Slug', 'statuspage-widget')}
                        value={slug}
                        onChange={handleSlugChange}
                        help={
                            slugError
                                ? slugError
                                : slug
                                    ? `URL: ${getStatusPageUrl(slug)}`
                                    : __('Your StatusPage.me slug (e.g., "your-company")', 'statuspage-widget')
                        }
                    />

                    <SelectControl
                        label={__('Theme', 'statuspage-widget')}
                        value={theme}
                        options={themeOptions}
                        onChange={(value) => setAttributes({ theme: value })}
                    />

                    <SelectControl
                        label={__('Render Mode', 'statuspage-widget')}
                        value={renderMode}
                        options={renderModeOptions}
                        onChange={(value) => setAttributes({ renderMode: value })}
                        help={
                            renderMode === 'js'
                                ? __('JavaScript mode supports auto-refresh and latest styling.', 'statuspage-widget')
                                : __('PHP mode is better for SEO but does not support auto-refresh.', 'statuspage-widget')
                        }
                    />
                </PanelBody>

                <PanelBody title={__('Display Options', 'statuspage-widget')} initialOpen={false}>
                    <ToggleControl
                        label={__('Compact Mode', 'statuspage-widget')}
                        checked={compact}
                        onChange={(value) => setAttributes({ compact: value })}
                        help={__('Display a smaller, inline version of the widget.', 'statuspage-widget')}
                    />

                    <ToggleControl
                        label={__('Show Active Incidents', 'statuspage-widget')}
                        checked={showIncidents}
                        onChange={(value) => setAttributes({ showIncidents: value })}
                    />

                    <ToggleControl
                        label={__('Disable Link to Status Page', 'statuspage-widget')}
                        checked={disableLink}
                        onChange={(value) => setAttributes({ disableLink: value })}
                    />
                </PanelBody>

                <PanelBody title={__('Sparklines', 'statuspage-widget')} initialOpen={false}>
                    <RangeControl
                        label={__('Response Time (minutes)', 'statuspage-widget')}
                        value={rtm}
                        onChange={(value) => setAttributes({ rtm: value })}
                        min={0}
                        max={240}
                        help={__('Set to 0 to hide the response time sparkline.', 'statuspage-widget')}
                    />

                    <RangeControl
                        label={__('Uptime (days)', 'statuspage-widget')}
                        value={upd}
                        onChange={(value) => setAttributes({ upd: value })}
                        min={0}
                        max={30}
                        help={__('Set to 0 to hide the uptime sparkline.', 'statuspage-widget')}
                    />
                </PanelBody>

                {renderMode === 'js' && (
                    <PanelBody title={__('Auto Refresh', 'statuspage-widget')} initialOpen={false}>
                        <RangeControl
                            label={__('Refresh Interval (seconds)', 'statuspage-widget')}
                            value={refresh / 1000}
                            onChange={(value) => setAttributes({ refresh: value * 1000 })}
                            min={0}
                            max={300}
                            step={10}
                            help={__('Set to 0 to disable auto-refresh. Minimum 10 seconds when enabled.', 'statuspage-widget')}
                        />
                    </PanelBody>
                )}
            </InspectorControls>

            <div {...blockProps}>
                {renderPreview()}
            </div>
        </>
    );
}
